#!/bin/bash
# =============================================================================
# Full Disk Backup & Restore (CIFS/NFS) con supporto SD Raspberry Pi
# =============================================================================
set -euo pipefail

### === CONFIGURAZIONE NAS ===
BACKUP_DIR="/mnt/backup"
NAS_TYPE="cifs"                             # cifs oppure nfs
NAS_PATH="//192.168.1.35/share/backup_pc"
SMB_CREDENTIALS_FILE="/root/.smbcredentials"
NAS_OPTS_CIFS_BASE="vers=3.0,nounix,noserverino"
NAS_OPTS_CIFS_EXTRA="uid=$(id -u),gid=$(id -g),file_mode=0640,dir_mode=0750"
NAS_OPTS_NFS="defaults"

### === BACKUP / RESTORE ===
DEVICE="${DEVICE:-}"                        # autodetect se vuoto
RETENTION_DAYS="${RETENTION_DAYS:-7}"

COMPRESS_BIN="$(command -v pigz || command -v gzip)"
COMPRESS_EXT="gz"

DATE="$(date +%Y%m%d-%H%M)"
HOSTNAME="$(hostname -s)"
BACKUP_BASENAME="${HOSTNAME}-backup-${DATE}.img.${COMPRESS_EXT}"
BACKUP_FILE="${BACKUP_DIR}/${BACKUP_BASENAME}"
CHECKSUM_FILE="${BACKUP_FILE}.sha256"

LOG_FILE="/var/log/full-disk-backup.log"

### === FUNZIONI BASE ===
log() { echo "[$(date '+%F %T')] $*" | tee -a "$LOG_FILE" >&2; }
require_root_or_sudo() {
  if [[ $EUID -ne 0 ]] && ! command -v sudo >/dev/null 2>&1; then
    echo "[ERRORE] Serve root o sudo." >&2; exit 1
  fi
}
run() { if [[ $EUID -ne 0 ]]; then sudo bash -c "$*"; else bash -c "$*"; fi; }
is_mounted() { mountpoint -q "$1"; }

### === NAS ===
mount_nas() {
  if is_mounted "$BACKUP_DIR"; then
    log "[*] NAS già montato su $BACKUP_DIR"
    return
  fi
  log "[*] Montaggio NAS $NAS_PATH su $BACKUP_DIR..."
  run "mkdir -p '$BACKUP_DIR'"

  if [[ "$NAS_TYPE" == "cifs" ]]; then
    if [[ -f "$SMB_CREDENTIALS_FILE" ]]; then
      CIFS_CREDS="credentials=$SMB_CREDENTIALS_FILE"
    else
      CIFS_CREDS="guest"
    fi
    run "mount -t cifs '$NAS_PATH' '$BACKUP_DIR' -o ${NAS_OPTS_CIFS_BASE},${NAS_OPTS_CIFS_EXTRA},${CIFS_CREDS}"
  else
    run "mount -t nfs '$NAS_PATH' '$BACKUP_DIR' -o $NAS_OPTS_NFS"
  fi

  if ! is_mounted "$BACKUP_DIR"; then
    log "[ERRORE] Impossibile montare NAS"; exit 1
  fi
}

### === DEVICE ===
autodetect_device() {
  local rootdev parent
  rootdev="$(findmnt -n -o SOURCE / || true)"
  if [[ -z "$rootdev" ]]; then echo ""; return; fi
  if [[ "$rootdev" =~ ^/dev/mmcblk[0-9]+p[0-9]+$ ]]; then
    parent="${rootdev%p*}"
  elif [[ "$rootdev" =~ ^/dev/sd[a-z][0-9]+$ ]]; then
    parent="${rootdev%[0-9]*}"
  elif [[ "$rootdev" =~ ^/dev/nvme[0-9]+n[0-9]+p[0-9]+$ ]]; then
    parent="${rootdev%p*}"
  else
    parent="$rootdev"
  fi
  echo "$parent"
}

check_device() {
  if [[ -z "${DEVICE}" || ! -b "${DEVICE}" ]]; then
    DEVICE="$(autodetect_device)"
  fi
  if [[ -z "${DEVICE}" || ! -b "${DEVICE}" ]]; then
    log "[ERRORE] Device non valido. Usa: DEVICE=/dev/xxx $0 backup"; exit 1
  fi
}

ensure_device_unmounted() {
  local dev="$1" parts
  parts="$(lsblk -nrpo NAME "$dev" | tail -n +2 || true)"
  if [[ -n "$parts" ]]; then
    while read -r p; do
      if mount | grep -q "^$p "; then
        log "[ERRORE] La partizione $p è montata. Smontala prima."; exit 1
      fi
    done <<< "$parts"
  fi
}

### === CHECKSUM ===
make_checksum() {
  log "[*] Calcolo SHA256: $(basename "$BACKUP_FILE")"
  (cd "$BACKUP_DIR" && sha256sum "$(basename "$BACKUP_FILE")") \
    | run "tee '$CHECKSUM_FILE' >/dev/null"
}

verify_checksum_file() {
  local img="$1" sum="${img}.sha256"
  [[ -f "$sum" ]] || { log "[!] Nessun checksum trovato"; return 1; }
  (cd "$(dirname "$img")" && sha256sum -c "$(basename "$sum")")
}

### === HOT BACKUP PREP ===
hot_freeze_fs() {
  log "[*] Sync e remount RO per backup a caldo..."
  sync
  if mount | grep -q "on / type"; then run "mount -o remount,ro /" || true; fi
  if mount | grep -q "on /boot type"; then run "mount -o remount,ro /boot" || true; fi
}
hot_unfreeze_fs() {
  log "[*] Ripristino RW..."
  if mount | grep -q "on / type"; then run "mount -o remount,rw /" || true; fi
  if mount | grep -q "on /boot type"; then run "mount -o remount,rw /boot" || true; fi
}

### === BACKUP ===
do_backup() {
  mount_nas
  check_device
  local size="$(lsblk -dn -o SIZE "$DEVICE")"
  log "[*] Avvio backup di $DEVICE ($size) -> $BACKUP_FILE"
  hot_freeze_fs

  set +e
  dd if="$DEVICE" bs=4M iflag=fullblock status=progress \
    | "$COMPRESS_BIN" -c \
    | run "tee '$BACKUP_FILE' >/dev/null"
  rc=${PIPESTATUS[0]}
  set -e

  hot_unfreeze_fs
  [[ $rc -eq 0 ]] || { log "[ERRORE] Backup fallito."; exit 1; }

  sync
  log "[+] Backup completato: $BACKUP_FILE"
  make_checksum
  cleanup_old_backups
}

cleanup_old_backups() {
  log "[*] Pulizia backup vecchi di $RETENTION_DAYS giorni..."
  find "$BACKUP_DIR" -type f -name "${HOSTNAME}-backup-*.img.${COMPRESS_EXT}" -mtime +$RETENTION_DAYS -exec rm -f {} {}.sha256 \;
}

### === RESTORE ===
list_backups() {
  mount_nas
  ls -1t "${BACKUP_DIR}/${HOSTNAME}-backup-*.img.${COMPRESS_EXT}" 2>/dev/null || true
}

do_restore() {
  mount_nas
  check_device
  ensure_device_unmounted "$DEVICE"

  mapfile -t BACKUPS < <(list_backups)
  local COUNT="${#BACKUPS[@]}"
  (( COUNT > 0 )) || { log "[ERRORE] Nessun backup trovato."; exit 1; }

  echo "Backup disponibili:"
  for i in "${!BACKUPS[@]}"; do echo "  [$i] ${BACKUPS[$i]}"; done
  echo ""
  read -r -p "Seleziona il numero del backup da ripristinare: " CHOICE
  [[ "$CHOICE" =~ ^[0-9]+$ && $CHOICE -lt $COUNT ]] || { log "[ERRORE] Scelta non valida."; exit 1; }

  SELECTED="${BACKUPS[$CHOICE]}"
  verify_checksum_file "$SELECTED" || {
    read -r -p "Checksum non valido/mancante. Procedere? (yes/no): " ok
    [[ "$ok" == "yes" ]] || exit 1
  }

  echo "ATTENZIONE: verrà SOVRASCRITTO TUTTO $DEVICE"
  read -r -p "Confermi? (yes/no): " confirm
  [[ "$confirm" == "yes" ]] || exit 0

  if [[ "$COMPRESS_BIN" =~ pigz$ ]]; then DECOMP="pigz -dc"; else DECOMP="gzip -dc"; fi
  run "dd if=/dev/zero of='$DEVICE' bs=1M count=10 conv=fsync"
  $DECOMP "$SELECTED" | run "dd of='$DEVICE' bs=4M status=progress conv=fsync"
  sync
  log "[+] Restore completato. Puoi riavviare con: sudo reboot"
}

### === MENU ===
menu() {
  clear
  echo "===================================================="
  echo "  BACKUP & RESTORE DISCO - ${HOSTNAME}"
  echo "===================================================="
  echo "1) Esegui Backup"
  echo "2) Esegui Restore Interattivo"
  echo "3) Elenca backup"
  echo "4) Esci"
  echo "===================================================="
  read -r -p "Scelta: " CHOICE
  case "$CHOICE" in
    1) do_backup ;;
    2) do_restore ;;
    3) list_backups ;;
    4) exit 0 ;;
    *) echo "[!] Scelta non valida" ;;
  esac
}

require_root_or_sudo

LOCKFILE="/var/lock/$(basename "$0").lock"
exec 9>"$LOCKFILE" || true
flock -n 9 || { log "[!] Un'altra istanza è in esecuzione. Esco."; exit 0; }

CMD="${1:-menu}"
case "$CMD" in
  backup)   do_backup ;;
  list)     list_backups ;;
  restore)  do_restore ;;
  menu|*)   menu ;;
esac