[CmdletBinding()]
param(
    [string[]]$ComputerName = @($env:SERVER-NAME),
    [string]$OutputPath = "C:\Temp",
    [switch]$IncludeIssued = $true
)

$ProductTypeMap = @{ 2='Per Device'; 4='Per User' }
$ProductVersionMap = @{ 2='Windows Server 2008'; 4='Windows Server 2012'; 8='Windows Server 2016/2019'; 16='Windows Server 2022/2025' }
function Convert-DmtfDate([string]$d){ if([string]::IsNullOrWhiteSpace($d)){return $null}; try{ [Management.ManagementDateTimeConverter]::ToDateTime($d) }catch{ $null } }

$null = New-Item -ItemType Directory -Path $OutputPath -Force -ErrorAction SilentlyContinue
$ts = Get-Date -Format "yyyyMMdd_HHmmss"
$outFile = Join-Path $OutputPath ("RDS_CAL_Report_{0}_{1}.csv" -f ($ComputerName -join '_'), $ts)

$rows = New-Object System.Collections.Generic.List[object]

function Get-LsData {
    param([string]$Server,[string]$NsPrimary='root\cimv2',[string]$NsFallback='root\cimv2\TerminalServices')

    $packs = @()
    $issued = @()

    foreach($ns in @($NsPrimary,$NsFallback)){
        if(-not $packs){
            try { $packs = Get-CimInstance -ComputerName $Server -Namespace $ns -ClassName Win32_TSLicenseKeyPack -ErrorAction Stop } catch {}
        }
        if($IncludeIssued -and -not $issued){
            try { $issued = Get-CimInstance -ComputerName $Server -Namespace $ns -ClassName Win32_TSIssuedLicense -ErrorAction Stop } catch {}
        }
    }
    [pscustomobject]@{ Packs=$packs; Issued=$issued }
}

foreach ($server in $ComputerName) {
  Write-Host ("`n>> Server: {0}" -f $server) -ForegroundColor Cyan

  $data = Get-LsData -Server $server
  $packs = $data.Packs
  $issued = $data.Issued

  if(-not $packs -and -not $issued -and $IncludeIssued){
    Write-Warning ("   Nessun dato trovato su {0}. Verifica che sia il License Server giusto e riprova in PowerShell 64-bit." -f $server)
  }

  foreach ($p in $packs) {
    $exp  = Convert-DmtfDate $p.ExpirationDate
    $days = if($exp){ [int][Math]::Floor(($exp - (Get-Date)).TotalDays) } else { $null }
    $rows.Add([pscustomobject]@{
      Server=$server; RecordType='Pack'; KeyPackId=$p.KeyPackId; Description=$p.Description
      ProductType=$p.ProductType; ProductTypeText=$ProductTypeMap[$p.ProductType]
      ProductVersion=$p.ProductVersion; ProductVersionText=$ProductVersionMap[$p.ProductVersion]
      KeyPackType=$p.KeyPackType; TotalLicenses=$p.TotalLicenses; AvailableLicenses=$p.AvailableLicenses
      IssuedCount=($p.TotalLicenses - $p.AvailableLicenses)
      IssueDate=$null; ExpirationDate=$exp; DaysToExpire=$days; MachineName=$null; LicenseStatus=$null
    })
  }

  if ($IncludeIssued) {
    foreach ($i in $issued) {
      $issue = Convert-DmtfDate $i.IssueDate
      $exp2  = Convert-DmtfDate $i.ExpirationDate
      $days2 = if($exp2){ [int][Math]::Floor(($exp2 - (Get-Date)).TotalDays) } else { $null }
      $rows.Add([pscustomobject]@{
        Server=$server; RecordType='IssuedDevice'; KeyPackId=$i.KeyPackId; Description=$null
        ProductType=$null; ProductTypeText=$null; ProductVersion=$null; ProductVersionText=$null
        KeyPackType=$null; TotalLicenses=$null; AvailableLicenses=$null; IssuedCount=$null
        IssueDate=$issue; ExpirationDate=$exp2; DaysToExpire=$days2; MachineName=$i.MachineName; LicenseStatus=$i.LicenseStatus
      })
    }
  }
}

if ($rows.Count -gt 0) {
  $csv = $rows | Select-Object * | ConvertTo-Csv -NoTypeInformation
  $Utf8NoBom = New-Object System.Text.UTF8Encoding($false)
  [System.IO.File]::WriteAllLines($outFile, $csv, $Utf8NoBom)
  Write-Host ("`nReport creato: {0}" -f $outFile) -ForegroundColor Green
  $outFile
} else {
  Write-Warning "Nessun dato da esportare."
}