#!/bin/bash


echo "Benvenuto! Questo script configura il Raspberry Pi come access point con VPN tramite WireGuard."

# Funzione per chiedere i dati all'utente, con valori predefiniti
ask_user_input() {
    local prompt=$1
    local default_value=$2
    local var_name=$3
    local is_sensitive=${4:-false}  # Se è una password o un dato sensibile

    if $is_sensitive; then
        read -s -p "$prompt [$default_value]: " input_value
        echo ""
    else
        read -p "$prompt [$default_value]: " input_value
    fi

    # Usa il valore predefinito se l'utente non fornisce input
    eval $var_name='${input_value:-$default_value}'
}

# Chiedi all'utente i parametri con valori predefiniti
ask_user_input "Inserisci l'interfaccia Wi-Fi" "wlan1" INTERFACCIA_WIFI
ask_user_input "Inserisci l'indirizzo IP statico" "192.168.4.1/24" IP_RETE
ask_user_input "Inserisci l'indirizzo LAN" "192.168.4.0/24" ADR_LAN
ask_user_input "Inserisci il range DHCP" "192.168.4.200,192.168.4.225,255.255.255.0,24h" DHCP_RANGE
ask_user_input "Inserisci il nome dell'SSID" "MullVPN" SSID
ask_user_input "Inserisci la passphrase WPA (min. 10 caratteri)" "!P@55w0rd!" WPA_PASSPHRASE

# Verifica la lunghezza della passphrase WPA
if [[ ${#WPA_PASSPHRASE} -lt 10 ]]; then
    echo "La passphrase WPA deve avere almeno 10 caratteri. Interrompo lo script."
    exit 1
fi

CHANNEL="7"  # Canale Wi-Fi predefinito

# Aggiorna e installa i pacchetti necessari
echo "Aggiornamento del sistema e installazione dei pacchetti necessari..."
sudo apt update && sudo apt upgrade -y
sudo apt install vim dnsmasq hostapd dhcpcd5 iptables iproute2 resolvconf iptables-persistent curl wireguard-tools -y

# Configurazione di /etc/dhcpcd.conf
echo "Configurazione di /etc/dhcpcd.conf..."
sudo bash -c "cat > /etc/dhcpcd.conf" <<EOF
interface ${INTERFACCIA_WIFI}
    static ip_address=${IP_RETE}
    nohook wpa_supplicant
EOF
sudo usermod -aG netdev dhcpcd

# Riavvia l'interfaccia Wi-Fi
echo "Riavvio dell'interfaccia Wi-Fi ${INTERFACCIA_WIFI}..."
sudo ip link set ${INTERFACCIA_WIFI} down
sudo ip link set ${INTERFACCIA_WIFI} up

# Configurazione di dnsmasq
echo "Configurazione di dnsmasq..."
sudo mv /etc/dnsmasq.conf /etc/dnsmasq.conf.orig
sudo bash -c "cat > /etc/dnsmasq.conf" <<EOF
interface=${INTERFACCIA_WIFI}
dhcp-range=${DHCP_RANGE}
EOF

# Configurazione di hostapd
echo "Configurazione di hostapd..."
sudo bash -c "cat > /etc/hostapd/hostapd.conf" <<EOF
interface=${INTERFACCIA_WIFI}
driver=nl80211
ssid=${SSID}
hw_mode=g
channel=${CHANNEL}
wmm_enabled=0
macaddr_acl=0
auth_algs=1
ignore_broadcast_ssid=0
wpa=2
wpa_passphrase=${WPA_PASSPHRASE}
wpa_key_mgmt=WPA-PSK
rsn_pairwise=CCMP
EOF

# Specifica il file di configurazione per hostapd
sudo bash -c "echo 'DAEMON_CONF=\"/etc/hostapd/hostapd.conf\"' > /etc/default/hostapd"

# Configurazione di /etc/sysctl.conf per l'inoltro IP
echo "Abilitazione dell'inoltro IP..."
sudo bash -c "echo 'net.ipv4.ip_forward=1' >> /etc/sysctl.conf"
sudo sysctl -p

# Configura WireGuard
echo "Configurazione di WireGuard..."
sudo cp /tmp/file-wg-001.conf /etc/wireguard/mullvad-wg.conf
sudo chown root:root /etc/wireguard/mullvad-wg.conf
sudo chmod 600 /etc/wireguard/mullvad-wg.conf
sudo wg-quick up mullvad-wg
sudo systemctl enable wg-quick@mullvad-wg

# NAT per la rete Wi-Fi attraverso WireGuard
echo "Configurazione NAT per la rete Wi-Fi tramite WireGuard..."
sudo iptables -t nat -A POSTROUTING -s ${ADR_LAN} -o mullvad-wg -j MASQUERADE

# Consentire traffico dalla rete Wi-Fi verso l'interfaccia WireGuard
sudo iptables -A FORWARD -i ${INTERFACCIA_WIFI} -o mullvad-wg -j ACCEPT
sudo iptables -A FORWARD -i mullvad-wg -o ${INTERFACCIA_WIFI} -m state --state RELATED,ESTABLISHED -j ACCEPT


# Salva regole iptables
echo "Salvataggio delle regole di iptables..."
sudo sh -c "iptables-save > /etc/iptables/rules.v4"

# Riavvia i servizi
echo "Riavvio dei servizi necessari..."
sudo systemctl unmask hostapd.service
sudo systemctl restart dhcpcd
sudo systemctl restart dnsmasq
sudo systemctl restart hostapd

sudo systemctl enable dhcpcd
sudo systemctl enable dnsmasq
sudo systemctl enable hostapd.service
sudo systemctl enable resolvconf

# Mostra l'indirizzo IP pubblico
echo "Configurazione completata! Il tuo indirizzo IP pubblico è:"
curl -s ifconfig.me